(function (window, undefined) {
    function SmartXPathEngine() {
        const engine = this;
        if (!(engine instanceof SmartXPathEngine)) {
            return new SmartXPathEngine();
        }
    }

    SmartXPathEngine.prototype.generateSmartXpath = function smartXPathEngine$generateSmartXpath(elementXPath, xpathDnd, hint, settings) {
        const CONSTANTS = {
            ATTRIBUTES: {
                ID: 'id',
                DATA: 'data-',
            },
            DIALOG_ELEMENTS: {
                HANDLE: {
                    LABEL: 'Handle',
                },
            },
            ERROR_MESSAGES: {
                ERROR_FETCHING_FRAME_BY_XPATH: 'Frame for target element could not be found with provided XPath $XPATH. Reason: $ERROR',
                ERROR_FETCHING_TARGET_ELEMENT_BY_XPATH: 'Target element could not be found with provided XPath $XPATH. Reason: $ERROR',
                ELEMENT_NOT_FOUND: 'Not found element with provided XPath',
                INVALID_XPATH: 'Invalid XPath',
                XPATH_CAN_NOT_BE_GENERATED_EXTENDED: 'XPath could not be generated, please try different label or press Continue button to use absolute XPath',
                XPATH_RETURNS_SEVERAL_ELEMENTS: 'XPath returns several elements',
            },
            HINT: {
                ATTRIBUTES_LIST: [
                    'value',
                    'placeholder',
                    'name',
                    'title'
                ],
                DISTANCE: {
                    VERY_CLOSE: 5,
                    CLOSE: 25,
                    MIDDLE: 50,
                    FAR: 120,
                    VERY_FAR: 250,
                },
                REQUIRED_NUMBER_OF_HINTS: 5,
            },
            NAMESPACE_URI_DEFAULT: 'http://www.w3.org/1999/xhtml',
            REPLACE_STRINGS: {
                ERROR: '$ERROR',
                XPATH: '$XPATH',
            },
            TAGS: {
                HTML: 'html',
                IFRAME: 'iframe',
                RECORDER_BLOCK: 'recorderblock',
            },
            XPATH_AXES: {
                ANCESTOR: 'ancestor',
                DESCENDANT: 'descendant',
            },
            XPATH_HELPERS: {
                AXES_NODE_DIVIDER: '::',
                NODE_DIVIDER: '/',
                START_XPATH: '//',
            },
        };
        const subject7IdAttr = 'subject7';
        const dialogHintBlockId = 'subject7dlgH';
        const dialogLocatorBlockId = 'subject7dlgL';
        let anchors = [];
        let labels = [];

        const getElementByXpath = (xpath, targetDocument, isAbsoluteXpath) => {
            // PROOF-10955 -> Oracle JET FilePicker support
            if (isAbsoluteXpath && xpath.includes('oj-file-picker')) {
                xpath = trimOracleJetXpath(xpath);
            }

            let result = targetDocument.evaluate(xpath, targetDocument, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);
            if (result.invalidIteratorState) {
                throw {code: 2, message: CONSTANTS.ERROR_MESSAGES.INVALID_XPATH};
            }

            let elements = [];
            let element;
            while ((element = result.iterateNext()) !== null) {
                elements.push(element);
            }

            if (elements.length === 0) {
                throw {code: 1, message: CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND};
            } else {
                if (elements.length === 1) {
                    return elements[0];
                } else {
                    throw {code: 0, message: CONSTANTS.ERROR_MESSAGES.XPATH_RETURNS_SEVERAL_ELEMENTS, elements: elements};
                }
            }
        };

        // PROOF-10955 -> Oracle JET FilePicker support
        const trimOracleJetXpath = (xpath) => {
            const match = /\/oj-file-picker(\[\d+\])?/.exec(xpath);

            if (match) {
                const index = match.index;
                const length = match[0].length;
                return xpath.substring(0, index + length);
            }

            return xpath;
        }

        const buildXpathByAnchors = (element, hint, settings) => {
            if (!hint) {
                return {
                    xpath: null,
                    distance: null
                };
            }
            anchors = [];

            if (settings.maxDistance) {
                findChildAnchors(element, hint, 0, settings.maxDistance, [], settings);
                findParentAnchors(element, hint, 1, settings.maxDistance, settings);
            } else {
                findAnchors(element, hint, settings);
            }
            if (anchors.length === 0) {
                return {
                    xpath: null,
                    distance: null
                };
            }

            const sortedAnchors = anchors.sort((a, b) => a.distance - b.distance);
            anchors = sortedAnchors;

            let xpath = buildXpath(element, sortedAnchors[0]);
            xpath = verifyAndClarifyXPath(xpath, element);

            return {
                xpath,
                distance: xpath !== null ? sortedAnchors[0].distance : null
            };
        };

        const findAnchors = (element, hint, settings) => {
            findChildAnchors(element, hint, 0, CONSTANTS.HINT.DISTANCE.CLOSE, [], settings);
            findParentAnchors(element, hint, 1, CONSTANTS.HINT.DISTANCE.CLOSE, settings);
            if (anchors.length === 0) {
                findChildAnchors(element, hint, 0, CONSTANTS.HINT.DISTANCE.MIDDLE, [], settings);
                findParentAnchors(element, hint, 1, CONSTANTS.HINT.DISTANCE.MIDDLE, settings);
                if (anchors.length === 0) {
                    findChildAnchors(element, hint, 0, CONSTANTS.HINT.DISTANCE.FAR, [], settings);
                    findParentAnchors(element, hint, 1, CONSTANTS.HINT.DISTANCE.FAR, settings);
                    if (anchors.length === 0) {
                        findChildAnchors(element, hint, 0, CONSTANTS.HINT.DISTANCE.VERY_FAR, [], settings);
                        findParentAnchors(element, hint, 1, CONSTANTS.HINT.DISTANCE.VERY_FAR, settings);
                    }
                }
            }
        };

        const findChildAnchors = (element, hint, distance, maxDistance, skipNodes, settings) => {
            const anchor = getAnchor(element, hint, settings);
            if (anchor !== null && distance <= maxDistance) {
                anchor.distance = distance;
                anchors.push(anchor);
            }

            let childNodes = getChildNodes(element);
            for (let i = 0; i < childNodes.length; i++) {
                const child = childNodes[i];
                if (skipNodes && skipNodes.indexOf(child) !== -1) {
                    continue;
                }
                findChildAnchors(child, hint, distance + 1, maxDistance, skipNodes, settings);
            }
        };

        const findParentAnchors = (element, hint, distance, maxDistance, settings) => {
            const parent = getParentNode(element);
            if (parent === null || distance >= maxDistance) {
                return;
            }
            findChildAnchors(parent, hint, distance + 1, maxDistance, [element], settings);
            findParentAnchors(parent, hint, distance + 1, maxDistance, settings);
        };

        const getParentNode = element => {
            let parent = element.parentNode;

            if (parent && parent.nodeType === Node.DOCUMENT_FRAGMENT_NODE && parent.host) {
                parent = parent.host;
            }

            return parent;
        };

        const getChildNodes = element => {
            let shadowRoot = element.shadowRoot;

            return (shadowRoot && shadowRoot.nodeType === Node.DOCUMENT_FRAGMENT_NODE && shadowRoot.host === element) ?
                shadowRoot.childNodes :
                element.childNodes;
        };

        const getAnchor = (element, hint, settings) => {
            let anchor = null;
            let tagName = (element.nodeType === Node.TEXT_NODE ||
                element.nodeType === Node.COMMENT_NODE ||
                element.nodeType === Node.DOCUMENT_NODE ||
                element.nodeType === Node.DOCUMENT_FRAGMENT_NODE ||
                element.nodeType === Node.DOCUMENT_TYPE_NODE) ?
                getNodeName(element) :
                getTagName(element);

            if ((tagName === CONSTANTS.TAGS.HTML) || (tagName === CONSTANTS.TAGS.RECORDER_BLOCK)) {
                return null;
            }

            if (tagName === CONSTANTS.TAGS.IFRAME) {
                let idAttr = element.getAttribute(CONSTANTS.ATTRIBUTES.ID);
                if (idAttr && idAttr.includes(subject7IdAttr)) {
                    return null;
                }
            }

            if (element.nodeType === Node.ELEMENT_NODE) {
                anchor = buildAnchorByAttribute(element, hint, settings, CONSTANTS.HINT.ATTRIBUTES_LIST);
                if (anchor === null) {
                    anchor = buildAnchorByInnerText(element, hint, settings);
                }
            } else if (element.nodeType === Node.TEXT_NODE && element.data) {
                anchor = buildAnchorForTextNode(element, hint, settings);
            }

            if (anchor === null) {
                anchor = buildAnchorByDataAttribute(element, hint, settings);
                if (anchor === null) {
                    return null;
                }
            }

            let elementByXpath = verifyAndClarifyXPath(CONSTANTS.XPATH_HELPERS.START_XPATH + anchor.condition, element);
            if (elementByXpath === null) {
                return null;
            }
            return anchor;
        };

        const getTagName = element => {
            return element.tagName.toLowerCase();
        };

        const getNodeName = element => {
            return element.nodeName.toLowerCase();
        };

        const isSVGElement = element => {
            return element instanceof SVGElement;
        };

        const containsCustomNamespace = element => {
            const hasNamespaceURI = element.namespaceURI === CONSTANTS.NAMESPACE_URI_DEFAULT;
            const regex = /^(?!:)[^:]+:[^:]+(?<!:)$/;
            const hasColon = regex.test(getTagName(element));
            return !hasNamespaceURI || (hasNamespaceURI && hasColon);
        };

        const verifyAndClarifyXPath = (xpath, element) => {
            try {
                const foundedElement = getElementByXpath(xpath, targetDocument);
                if (element === foundedElement || isTheSameElements(element, foundedElement) ||
                    (isSVGElement(element) && element === getParentNode(foundedElement)) ||
                    (isSVGElement(foundedElement) && element === getChildNodes(foundedElement)[0])) {
                    return xpath;
                }
                return null;
            } catch (error) {
                if (error.code === 0 && error.elements.indexOf(element) !== -1) {
                    let elements = error.elements.filter(value => {
                        if (value.nodeType === Node.TEXT_NODE) {
                            return true;
                        } else {
                            let id = value.getAttribute(CONSTANTS.ATTRIBUTES.ID);
                            return id !== dialogLocatorBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.HANDLE.LABEL) &&
                                id !== dialogHintBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.HANDLE.LABEL);
                        }
                    });
                    return `(${xpath})[${elements.indexOf(element) + 1}]`;
                }
                return null;
            }
        };

        const isTheSameElements = (targetElem, elem) => {
            return (targetElem.baseURI === elem.baseURI) &&
                (targetElem.clientHeight === elem.clientHeight) &&
                (targetElem.clientWidth === elem.clientWidth) &&
                (targetElem.clientLeft === elem.clientLeft) &&
                (targetElem.clientTop === elem.clientTop) &&
                (targetElem.className === elem.className) &&
                (targetElem.contentEditable === elem.contentEditable) &&
                (targetElem.id === elem.id) &&
                (targetElem.innerHTML === elem.innerHTML) &&
                (targetElem.innerText === elem.innerText) &&
                (targetElem.isConnected === elem.isConnected) &&
                (targetElem.isContentEditable === elem.isContentEditable) &&
                (targetElem.localName === elem.localName) &&
                (targetElem.nodeName === elem.nodeName) &&
                (targetElem.nodeType === elem.nodeType) &&
                (targetElem.offsetHeight === elem.offsetHeight) &&
                (targetElem.offsetLeft === elem.offsetLeft) &&
                (targetElem.offsetTop === elem.offsetTop) &&
                (targetElem.offsetWidth === elem.offsetWidth) &&
                (targetElem.outerHTML === elem.outerHTML) &&
                (targetElem.outerText === elem.outerText) &&
                (targetElem.scrollHeight === elem.scrollHeight) &&
                (targetElem.scrollLeft === elem.scrollLeft) &&
                (targetElem.scrollTop === elem.scrollTop) &&
                (targetElem.scrollWidth === elem.scrollWidth) &&
                (targetElem.tagName === elem.tagName) &&
                (targetElem.textContent === elem.textContent) &&
                (targetElem.title === elem.title);
        };

        const buildAnchorByInnerText = (element, hint, settings) => {
            if (!element.innerText) {
                return null;
            }
            let innerText = element.innerText.replace(/\s/g, ' ');
            let sourceInnerText = innerText;
            let hintText = hint;

            if (settings.matchCase !== true) {
                innerText = innerText.toLocaleLowerCase();
                hintText = hintText.toLocaleLowerCase();
            }

            if (settings.fullMatch === true && innerText === hintText) {
                return {
                    element: element,
                    condition: isSVGElement(element) || containsCustomNamespace(element) ?
                        `*[name()='${element.localName}' and text()='${sourceInnerText}']` :
                        `${element.localName}[text()='${sourceInnerText}']`
                };
            } else {
                if (settings.fullMatch === false && innerText.indexOf(hintText) !== -1) {
                    let originValue = settings.isWildCard ?
                        sourceInnerText :
                        sourceInnerText.substr(innerText.indexOf(hintText), hintText.length);
                    return {
                        element: element,
                        condition: isSVGElement(element) || containsCustomNamespace(element) ?
                            `*[name()='${element.localName}' and contains(@text, '${originValue}')]` :
                            `${element.localName}[contains(text(),'${originValue}')]`
                    };
                }
            }
            return null;
        };

        const buildAnchorForTextNode = (element, hint, settings) => {
            if (!element.data) {
                return null;
            }

            let textOrigin = element.data.replace(/\s/g, ' ');
            let text = textOrigin;
            let hintText = hint;

            if (settings.matchCase !== true) {
                text = text.toLocaleLowerCase();
                hintText = hintText.toLocaleLowerCase();
            }

            let anchor = null;
            if (settings.fullMatch === true && text === hintText) {
                anchor = {
                    element: element,
                    condition: `text()[.='${textOrigin}']`
                };
            } else {
                if (settings.fullMatch === false && text.indexOf(hintText) !== -1) {
                    let originValue = settings.isWildCard ?
                        textOrigin :
                        textOrigin.substr(text.indexOf(hintText), hintText.length);
                    anchor = {
                        element: element,
                        condition: `text()[contains(.,'${originValue}')]`
                    };
                }
            }
            return anchor;
        };

        const buildAnchorByAttribute = (element, hint, settings, attributesList) => {
            for (let attribute of attributesList) {
                if (element.hasAttribute(attribute)) {
                    let attributeValue = element.getAttribute(attribute);
                    let value = hint;

                    if (settings.matchCase !== true) {
                        attributeValue = attributeValue.toLocaleLowerCase();
                        value = value.toLocaleLowerCase();
                    }

                    if (settings.fullMatch === true && attributeValue === value) {
                        return {
                            element: element,
                            condition: isSVGElement(element) || containsCustomNamespace(element) ?
                                `*[name()='${element.localName}' and @${attribute}='${element.getAttribute(attribute)}']` :
                                `${element.localName}[@${attribute}='${element.getAttribute(attribute)}']`
                        }
                    } else {
                        if (settings.fullMatch === false && attributeValue.indexOf(value) !== -1) {
                            let attr = element.getAttribute(attribute);
                            let originValue = settings.isWildCard ?
                                attr :
                                attr.substr(attributeValue.indexOf(value), value.length);
                            return {
                                element: element,
                                condition: isSVGElement(element) || containsCustomNamespace(element) ?
                                    `*[name()='${element.localName}' and contains(@${attribute}, '${originValue}')]` :
                                    `${element.localName}[contains(@${attribute}, '${originValue}')]`
                            }
                        }
                    }
                }
            }
            return null;
        };

        const buildAnchorByDataAttribute = (element, hint, settings) => {
            if (!element.dataset) {
                return null;
            }
            let attributesList = Object.keys(element.dataset)
                .map(key => CONSTANTS.ATTRIBUTES.DATA + camelToDash(key));
            return buildAnchorByAttribute(element, hint, settings, attributesList);
        };

        const camelToDash = str => {
            return str
                .replace(/([a-z0-9])([A-Z])/g, '$1-$2')
                .toLowerCase();
        };

        const buildXpath = (element, anchor) => {
            let hintElement = anchor.element;
            let xpath = CONSTANTS.XPATH_HELPERS.START_XPATH + anchor.condition;

            if (element === hintElement ||
                (isSVGElement(element) && element === getParentNode(hintElement)) ||
                (isSVGElement(hintElement) && element === getChildNodes(hintElement)[0])) {
                return xpath;
            }

            let commonElement = findCommonNode(element, hintElement);
            let pathToAnchor = getPathHierarchy(commonElement, hintElement);
            let pathToElement = getPathHierarchy(commonElement, element);

            if (pathToAnchor.length > 0) {
                xpath += CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                    CONSTANTS.XPATH_AXES.ANCESTOR +
                    CONSTANTS.XPATH_HELPERS.AXES_NODE_DIVIDER
                    + buildAdditionalExpression(commonElement);
            }

            if (pathToElement.length > 0) {
                let xpathPrefix = xpath +
                    CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                    CONSTANTS.XPATH_AXES.DESCENDANT +
                    CONSTANTS.XPATH_HELPERS.AXES_NODE_DIVIDER;
                let descendantHierarchy = [];
                for (let currentElement of pathToElement) {
                    descendantHierarchy.push(buildAdditionalExpression(currentElement));
                    let currentXpath = xpathPrefix + descendantHierarchy.slice(0).reverse().join(CONSTANTS.XPATH_HELPERS.NODE_DIVIDER);
                    try {
                        getElementByXpath(currentXpath, targetDocument);
                    } catch (error) {
                        if (error.code === 0 && error.elements.indexOf(element) !== -1 && error.elements.length <= 10) {
                            return currentXpath;
                        } else {
                            continue;
                        }
                    }
                    return currentXpath;
                }
                return xpath + CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                    pathToElement
                        .reverse()
                        .map(node => isSVGElement(node) || containsCustomNamespace(element) ? `*[name()='${node.localName}']` : node.localName)
                        .join(CONSTANTS.XPATH_HELPERS.NODE_DIVIDER);
            } else {
                return xpath;
            }
        };

        const buildAdditionalExpression = element => {
            return isSVGElement(element) || containsCustomNamespace(element) ? `*[name()='${element.localName}']` : element.localName;
        };

        const findCommonNode = (node1, node2) => {
            let parentNode1 = node1;
            while (getParentNode(parentNode1) !== null) {
                let parentNode2 = node2;
                while (getParentNode(parentNode2) !== null) {
                    if (parentNode1 === parentNode2) {
                        return parentNode1;
                    }
                    parentNode2 = getParentNode(parentNode2);
                }
                parentNode1 = getParentNode(parentNode1);
            }
            return null;
        };

        const getPathHierarchy = (fromElement, toElement) => {
            let currentElement = toElement;
            let path = [];
            while (currentElement !== null && currentElement !== fromElement) {
                path.push(currentElement);
                currentElement = getParentNode(currentElement);
            }
            return path;
        };

        let element;
        let targetDocument = document;

        const findLabelsForTarget = element => {
            labels = [];

            if (settings.maxDistance) {
                findChildLabels(element, 0, settings.maxDistance, []);
                findParentLabels(element, 1, settings.maxDistance);
            } else {
                findLabels(element);
            }

            return labels
                .sort((a, b) => a.distance - b.distance)
                .filter(label => label !== '')
                .filter(label => label.text.trim().length);
        };

        const findLabels = element => {
            findChildLabels(element, 0, CONSTANTS.HINT.DISTANCE.CLOSE, []);
            findParentLabels(element, 1, CONSTANTS.HINT.DISTANCE.CLOSE);
            if (labels.length === 0) {
                findChildLabels(element, 0, CONSTANTS.HINT.DISTANCE.MIDDLE, []);
                findParentLabels(element, 1, CONSTANTS.HINT.DISTANCE.MIDDLE);
                if (labels.length === 0) {
                    findChildLabels(element, 0, CONSTANTS.HINT.DISTANCE.FAR, []);
                    findParentLabels(element, 1, CONSTANTS.HINT.DISTANCE.FAR);
                    if (labels.length === 0) {
                        findChildLabels(element, 0, CONSTANTS.HINT.DISTANCE.VERY_FAR, []);
                        findParentLabels(element, 1, CONSTANTS.HINT.DISTANCE.VERY_FAR);
                    }
                }
            }
        };

        const findChildLabels = (element, distance, maxDistance, skipNodes) => {
            const label = getLabel(element);
            if (label !== null && label.trim().length !== 0 && distance <= maxDistance && labels.length <= CONSTANTS.HINT.REQUIRED_NUMBER_OF_HINTS) {
                labels.push({text: label, distance: distance});
            }

            let childNodes = getChildNodes(element);
            for (let i = 0; i < childNodes.length; i++) {
                const child = childNodes[i];
                if (skipNodes && skipNodes.indexOf(child) !== -1) {
                    continue;
                }
                findChildLabels(child, distance + 1, maxDistance, skipNodes);
            }
        };

        const findParentLabels = (element, distance, maxDistance) => {
            const parent = getParentNode(element);
            if (parent === null || distance >= maxDistance) {
                return;
            }
            findChildLabels(parent, distance + 1, maxDistance, [element]);
            findParentLabels(parent, distance + 1, maxDistance);
        };

        const getLabel = element => {
            let label = null;

            if (element.nodeType === Node.ELEMENT_NODE) {
                label = getLabelByAttribute(element, CONSTANTS.HINT.ATTRIBUTES_LIST);
                if (label === null) {
                    label = getLabelByInnerText(element);
                }
            } else if (element.nodeType === Node.TEXT_NODE && element.data) {
                label = getLabelForTextNode(element);
            }

            if (label === null) {
                label = getLabelByDataAttribute(element);
                if (label === null) {
                    return null;
                }
            }

            return label;
        };

        const getLabelByInnerText = element => {
            return element.innerText ?
                element.innerText.replace(/\s/g, ' ') :
                null;
        };

        const getLabelForTextNode = element => {
            return element.data ?
                element.data.replace(/\s/g, ' ') :
                null;
        };

        const getLabelByAttribute = (element, attributesList) => {
            for (let attribute of attributesList) {
                if (element.hasAttribute(attribute)) {
                    return element.getAttribute(attribute);
                }
            }
            return null;
        };

        const getLabelByDataAttribute = element => {
            if (!element.dataset) {
                return null;
            }

            let attributesList = Object.keys(element.dataset)
                .map(key => CONSTANTS.ATTRIBUTES.DATA + camelToDash(key));
            return getLabelByAttribute(element, attributesList);
        };

        try {
            element = getElementByXpath(xpathDnd ? xpathDnd : elementXPath, targetDocument, true);
        } catch (error) {
            return {
                xpath: elementXPath,
                xpathDnd: xpathDnd,
                hint: hint,
                error: CONSTANTS.ERROR_MESSAGES.ERROR_FETCHING_TARGET_ELEMENT_BY_XPATH
                    .replace(CONSTANTS.REPLACE_STRINGS.XPATH, xpathDnd ? xpathDnd : elementXPath)
                    .replace(CONSTANTS.REPLACE_STRINGS.ERROR, error.message)
            }
        }

        let result = [];
        let hints;
        let isWildCard = hint !== null ? hint.startsWith('*') : false;

        if (!settings) {
            settings = {
                maxDistance: CONSTANTS.HINT.DISTANCE.CLOSE,
                matchCase: false,
                fullMatch: false,
                useIdAttribute: false,
                isWildCard: isWildCard
            };
        } else {
            settings.isWildCard = isWildCard;
        }

        if (settings.useIdAttribute && element.hasAttribute(CONSTANTS.ATTRIBUTES.ID)) {
            let attrValue = element.getAttribute(CONSTANTS.ATTRIBUTES.ID);

            return {
                xpath: isSVGElement(element) || containsCustomNamespace(element) ?
                    `//*[name()='${element.localName}'[@id='${attrValue}']` :
                    `//${element.localName}[@id='${attrValue}']`,
                xpathDnd: xpathDnd,
                hint: hint !== null ? hint : null,
                error: null
            };
        }

        if (hint !== null) {
            hints = [isWildCard ? {text: hint.slice(1)} : {text: hint}];
        } else {
            hints = [...new Set(findLabelsForTarget(element))];
        }

        hints.forEach(hint => {
            result.push(buildXpathByAnchors(element, hint.text, settings));
        });

        let sortedResult = result
            .filter(item => item.xpath !== null)
            .sort((a, b) => a.distance - b.distance);

        if (!sortedResult[0]) {
            return {
                xpath: elementXPath,
                xpathDnd: xpathDnd,
                hint: hints.length !== 0 ? hints[0].text : null,
                error: CONSTANTS.ERROR_MESSAGES.XPATH_CAN_NOT_BE_GENERATED_EXTENDED
            };
        }

        return {
            xpath: sortedResult[0].xpath === null ? elementXPath : sortedResult[0].xpath,
            xpathDnd: xpathDnd,
            hint: hints.length !== 0 ? hints[0].text : null,
            error: sortedResult[0].xpath === null ? CONSTANTS.ERROR_MESSAGES.XPATH_CAN_NOT_BE_GENERATED_EXTENDED : null
        };
    }

    window.smartXPathEngine = new SmartXPathEngine();
})(window);
