let targetTabId = null;
let welcomeDialogTabId = null;
let responseReceived = false;
let isNavigatorWebdriver = false;
let dialogTabId = null;
let logInfoDialogTabId = null;
let logInfoPort = null;
let LOG_INFO = [];

importScripts(
    'constants.js',
    'services/communication.service.js',
    'services/events.service.js',
    'services/apiService.js'
);

const keepServiceRunning = () => {
    setTimeout(keepServiceRunning, CONSTANTS.KEEP_ALIVE_CHECK_TIMEOUT);
};

keepServiceRunning();
ApiService();

chrome.storage.local.set({
    locator: {},
    hint: {
        text: '',
        maxDistance: 25,
        matchCase: false,
        fullMatch: false,
        useIdAttribute: false
    }
});

const deleteTimer = port => {
    if (port._timer) {
        clearTimeout(port._timer);
        delete port._timer;
    }
};

chrome.contextMenus.onClicked.addListener((info, tab) => {
    if (info.menuItemId === CONSTANTS.LOG_INFO.MENU_ID) {
        createLogInfoPopup();
    }
});

chrome.runtime.onConnect.addListener(portObject => {
    switch (portObject.name) {
        case CONSTANTS.MESSAGING_PORT:
            portObject.onMessage.addListener(async (msg, port) => {
                let data;

                isNavigatorWebdriver = msg.isAutomationBrowser;

                if (!isNavigatorWebdriver) {
                    data = await chrome.storage.local.get(['locatorExplorerInitFlag', 'saveFeatureEnabled']);

                    if (!data.locatorExplorerInitFlag) {
                        await chrome.storage.local.set({locatorExplorerInitFlag: true});
                        openWelcomeDialog();
                        removeLogMenuItem(true);
                        if (data.saveFeatureEnabled) {
                            addLogMenuItem();
                        }
                    } else if (data.saveFeatureEnabled) {
                        createOrUpdateLogMenuItem();
                    }
                } else {
                    data = await chrome.storage.local.get('getConnectionInfoInitFlag');

                    if (!data.getConnectionInfoInitFlag) {
                        await chrome.storage.local.set({getConnectionInfoInitFlag: true});
                        CommunicationService(isNavigatorWebdriver);
                        setTimeout(CommunicationService.sendConnectionRequest, 250);
                        addLogMenuItem();
                    }
                }
            });
            portObject.onDisconnect.addListener(deleteTimer);
            portObject._timer = setTimeout(
                port => {
                    deleteTimer(port);
                    port.disconnect();
                },
                CONSTANTS.KEEP_ALIVE_CHECK_TIMEOUT,
                portObject
            );
            break;
        case CONSTANTS.LOG_INFO.PORT:
            logInfoPort = portObject;

            console.debug(`${new Date(Date.now()).toISOString()} -> [LOG_INFO] Port '${logInfoPort.name}' is open and connected`);
            sendMessageToLogInfo({records: LOG_INFO});

            logInfoPort.onMessage.addListener((msg, port) => {
                if (msg) {
                    if (msg.clearLog) {
                        LOG_INFO = [];
                        sendMessageToLogInfo({records: LOG_INFO});
                    }
                }
            });

            logInfoPort.onDisconnect.addListener(() => {
                logInfoPort = null;
                logInfoDialogTabId = null;
            });
            break;
        default:
            break;
    }
});

chrome.runtime.onMessage.addListener(async (request, sender, sendResponse) => {
    let action = request.action;
    switch (action) {
        case CONSTANTS.ACTIONS.ELEMENT_CAPTURED:
            let {isFrameset, isDisabledHint, isDisabledGenerating} = request;
            const data = await chrome.storage.local.get('hint');

            if (data.hint) {
                data.hint.text = '';
                await chrome.storage.local.set({hint: data.hint});
                if (isFrameset) {
                    await chrome.storage.local.set({isFrameset});
                    openHintDialog();
                } else {
                    await openHintFrameDialog({
                        hint: data.hint,
                        isFrameset,
                        isDisabledHint,
                        isDisabledGenerating
                    });
                }
            } else {
                if (isFrameset) {
                    await chrome.storage.local.set({hint: null, isFrameset});
                    openHintDialog();
                } else {
                    await openHintFrameDialog({
                        hint: null,
                        isFrameset,
                        isDisabledHint,
                        isDisabledGenerating
                    });
                }
            }
            break;
        case CONSTANTS.ACTIONS.HINT_DIALOG_RESULT:
            processHintDialogResult(request);
            break;
        case CONSTANTS.ACTIONS.LOCATOR_DIALOG_RESULT:
            await processLocatorDialogResult(request.result);
            break;
        case CONSTANTS.ACTIONS.TEST_XPATH_FINISHED:
            chrome.tabs.sendMessage(targetTabId, {
                action: CONSTANTS.ACTIONS.TEST_CSS,
                css: request.css,
                isFrameset: request.isFrameset
            });
            break;
        case CONSTANTS.ACTIONS.TEST_CSS_FINISHED:
            if (request.isFrameset) {
                focusTab(dialogTabId);
            } else {
                chrome.tabs.sendMessage(targetTabId, {
                    action: CONSTANTS.ACTIONS.SHOW_DIALOG,
                    dialogType: CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE
                });
            }
            break;
        case CONSTANTS.ACTIONS.GENERATION_FINISHED:
            if (request.result.hasError) {
                if (request.isFrameset) {
                    await chrome.storage.local.set({hint: request.hint, isFrameset: request.isFrameset});
                    openHintDialog();
                } else {
                    await openHintFrameDialog({
                        hint: request.hint,
                        isFrameset: request.isFrameset,
                        isDisabledHint: false,
                        algorithmType: null
                    });
                }
            } else {
                const data = await chrome.storage.local.get('locator');

                let locator = {};
                if (data.locator) {
                    locator = data.locator;
                }
                locator.xpath = request.result.xpath;
                locator.cssSelector = request.result.css;
                locator.frame = request.result.frame;
                locator.popup = '';
                await chrome.storage.local.set({locator});
                if (request.isFrameset) {
                    await chrome.storage.local.set({hint: request.hint, isFrameset: request.isFrameset});
                    await openLocatorDialog();
                } else {
                    await openLocatorFrameDialog({
                        locator,
                        hint: request.hint,
                        isFrameset: request.isFrameset,
                        isDisabledHint: request.isDisabledHint,
                        algorithmType: request.result.algorithmType
                    });
                }
            }
            break;
        case CONSTANTS.ACTIONS.CLOSE_WELCOME_DIALOG:
            closeWelcomeDialog();
            break;
        case CONSTANTS.ACTIONS.SYNC_DIALOG_SIZE: {
            chrome.tabs.sendMessage(targetTabId, request);
            break;
        }
        case CONSTANTS.ACTIONS.SEND_CHROME_NOTIFICATION:
            showChromeNotification(request.message);
            break;
        case CONSTANTS.ACTIONS.SEND_NOTIFICATION_MESSAGE:
            const {message, type} = request;
            sendNotification(message, type);
            break;
        case CONSTANTS.ACTIONS.SYNC_NOTIFICATION_SIZE: {
            chrome.tabs.sendMessage(targetTabId, request);
            break;
        }
        case CONSTANTS.ACTIONS.CLOSE_NOTIFICATION:
            closeNotification(request.type);
            break;
        case CONSTANTS.ACTIONS.SEND_LOG_INFO:
            let logInfo = {
                logInfo: request.logInfo,
                element: request.element,
                hint: request.hint,
                additionalInfo: request.additionalInfo
            };
            LOG_INFO.push(logInfo);
            sendMessageToLogInfo({record: logInfo});
            break;
        case CONSTANTS.ACTIONS.ADD_LOG_MENU:
            createOrUpdateLogMenuItem();
            break;
        case CONSTANTS.ACTIONS.REMOVE_LOG_MENU:
            removeLogMenuItem(true);
            break;
        default:
            break;
    }
});

chrome.runtime.onMessageExternal.addListener((request, sender, sendResponse) => {
    if (sender.id === CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.ID || sender.id === CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.ID_FROM_STORE) {
        if (request.reason === CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.REASON) {
            if (targetTabId) {
                stopCapturing(true);
            }
            sendResponse();
            return true;
        }
    }
    sendResponse();
    return true;
});

chrome.tabs.onActivated.addListener((event) => {
    if (event.tabId !== dialogTabId && event.tabId !== logInfoDialogTabId) {
        setTimeout(() => {
            changeIcon(targetTabId && (event.tabId === targetTabId));
        }, 500);
    }
});

chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
    if (changeInfo.status === 'complete' && targetTabId && tabId === targetTabId) {
        chrome.tabs.sendMessage(tabId, {
            action: CONSTANTS.ACTIONS.START_CAPTURING
        });
    }
});

chrome.tabs.onRemoved.addListener((tabId) => {
    if (targetTabId && tabId === targetTabId) {
        targetTabId = null;
        changeIcon(false);
    }
});

chrome.action.onClicked.addListener((tab) => {
    responseReceived = false;

    closeWelcomeDialog();

    if (targetTabId && tab.id !== targetTabId) {
        chrome.tabs.sendMessage(targetTabId, {action: CONSTANTS.ACTIONS.STOP_CAPTURING}, () => {
            changeIcon(false);
            startCapturing(tab.id);
        });
    } else {
        if (targetTabId) {
            stopCapturing(false);
        } else {
            chrome.management.get(
                CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.ID,
                answer => {
                    if (!chrome.runtime.lastError) {
                        chrome.runtime.sendMessage(
                            CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.ID,
                            {
                                reason: CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.REASON
                            },
                            null,
                            response => {
                                processExtensionResponse(response, tab.id);
                            }
                        );
                    } else {
                        chrome.runtime.sendMessage(
                            CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.ID_FROM_STORE,
                            {
                                reason: CONSTANTS.EXTERNAL_EXTENSIONS.RECORDER.REASON
                            },
                            null,
                            response => {
                                processExtensionResponse(response, tab.id);
                            }
                        );
                    }
                }
            )
        }
    }
    setTimeout(() => {
        if (!responseReceived) {
            showMessage(CONSTANTS.NOTIFICATION_MESSAGES.REFRESH_PAGE);
        }
    }, 1500);
});

const processExtensionResponse = (response, tabId) => {
    if (!chrome.runtime.lastError) {
        if (response) {
            if (response.isEnabled) {
                responseReceived = true;
                showMessage(CONSTANTS.NOTIFICATION_MESSAGES.CANNOT_ENABLE_GUTENBERG);
            } else {
                startCapturing(tabId);
            }
        } else {
            startCapturing(tabId);
        }
    } else {
        startCapturing(tabId);
    }
};

const startCapturing = tabId => {
    chrome.tabs.sendMessage(tabId, {action: CONSTANTS.ACTIONS.START_CAPTURING}, response => {
        responseReceived = true;
        if (!chrome.runtime.lastError) {
            if (!response || !response.started) {
                showMessage(CONSTANTS.NOTIFICATION_MESSAGES.REFRESH_PAGE);
                return;
            }
            targetTabId = tabId;
            changeIcon(true);
        } else {
            showMessage(CONSTANTS.NOTIFICATION_MESSAGES.REFRESH_PAGE);
        }
    });
};

const stopCapturing = showNotification => {
    chrome.tabs.sendMessage(targetTabId, {action: CONSTANTS.ACTIONS.STOP_CAPTURING}, response => {
        responseReceived = true;
        if (!chrome.runtime.lastError) {
            if (!response || response.started) {
                showMessage(CONSTANTS.NOTIFICATION_MESSAGES.REFRESH_PAGE);
                return;
            }
            targetTabId = null;
            changeIcon(false);
            if (showNotification) {
                showMessage(CONSTANTS.NOTIFICATION_MESSAGES.GUTENBERG_IS_INACTIVE);
            }
        } else {
            showMessage(CONSTANTS.NOTIFICATION_MESSAGES.REFRESH_PAGE);
        }
    });
};

const changeIcon = enabled => {
    chrome.action.setIcon({
        path: enabled ?
            '/img/app-icon128x128.png' :
            '/img/app-disabled.png'
    });
    chrome.action.setTitle({
        title: enabled ?
            CONSTANTS.EXTENSION_NAME.concat(' Enabled') :
            'Click to enable '.concat(CONSTANTS.EXTENSION_NAME)
    });
};

const processHintDialogResult = request => {
    if (request.isFrameset) {
        closeDialog();
    } else {
        closeFrameDialog(CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE);
    }
    if (request.hint) {
        chrome.tabs.sendMessage(targetTabId, {
            action: CONSTANTS.ACTIONS.GENERATE_XPATH,
            hint: request.hint,
            isDisabledHint: request.isDisabledHint
        });
    } else {
        chrome.tabs.sendMessage(targetTabId, {
            action: CONSTANTS.ACTIONS.RESET_CAPTURING,
            fireEvent: request.skip
        });
    }
};

const processLocatorDialogResult = async result => {
    switch (result.type) {
        case CONSTANTS.ACTIONS.BACK_TO_HINT:
            if (result.isFrameset) {
                closeDialog();
                await chrome.storage.local.set({hint: result.hint, isFrameset: result.isFrameset});
                openHintDialog();
            } else {
                closeFrameDialog(CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE);
                await openHintFrameDialog({
                    hint: result.hint,
                    isFrameset: result.isFrameset,
                    isDisabledHint: result.isDisabledHint
                });
            }
            break;
        case CONSTANTS.ACTIONS.TEST:
            if (result.isFrameset) {
                focusTab(targetTabId);
            } else {
                chrome.tabs.sendMessage(targetTabId, {
                    action: CONSTANTS.ACTIONS.HIDE_DIALOG,
                    dialogType: CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE
                });
            }
            chrome.tabs.sendMessage(targetTabId, {
                action: CONSTANTS.ACTIONS.TEST_XPATH,
                xpath: result.xpath,
                css: result.cssSelector,
                isFrameset: result.isFrameset
            });
            break;
        case CONSTANTS.ACTIONS.SKIP:
            if (result.isFrameset) {
                closeDialog();
            } else {
                closeFrameDialog(CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE);
            }
            chrome.tabs.sendMessage(targetTabId, {
                action: CONSTANTS.ACTIONS.RESET_CAPTURING,
                fireEvent: result.skip
            });
            break;
        default:
            break;
    }
};

const openWelcomeDialog = () => {
    welcomeDialogTabId = null;

    chrome.windows.getCurrent(window => {
        let screenDimensions =
            {
                width: window.width,
                height: window.height
            };
        chrome.windows.create({
                url: CONSTANTS.DIALOG_OPTIONS.WELCOME.URL,
                type: 'popup',
                width: CONSTANTS.DIALOG_OPTIONS.WELCOME.WIDTH,
                height: CONSTANTS.DIALOG_OPTIONS.WELCOME.HEIGHT,
                left: Math.round((screenDimensions.width / 2) - (CONSTANTS.DIALOG_OPTIONS.WELCOME.WIDTH / 2)),
                top: Math.round((screenDimensions.height / 2) - (CONSTANTS.DIALOG_OPTIONS.WELCOME.HEIGHT / 2)),
                focused: false
            },
            window => {
                welcomeDialogTabId = window.tabs[0].id;
                focusTab(welcomeDialogTabId);
            });
    });
};

const closeWelcomeDialog = () => {
    if (welcomeDialogTabId !== null) {
        chrome.tabs.get(welcomeDialogTabId, (tab) => {
            welcomeDialogTabId = null;
            chrome.windows.remove(tab.windowId);
        });
    }
};

const openHintFrameDialog = async options => {
    const data = await chrome.storage.local.get('saveFeatureEnabled');

    options.saveFeatureEnabled = data.saveFeatureEnabled;
    openFrameDialog(CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE, options);
};

const openLocatorFrameDialog = async options => {
    const data = await chrome.storage.local.get('saveFeatureEnabled');

    options.saveFeatureEnabled = data.saveFeatureEnabled;
    openFrameDialog(CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE, options);
};

const openFrameDialog = (dialogType, options) => {
    chrome.tabs.sendMessage(targetTabId, {
        action: CONSTANTS.ACTIONS.OPEN_DIALOG,
        dialogType,
        options
    });
};

const closeFrameDialog = dialogType => {
    chrome.tabs.sendMessage(targetTabId, {
        action: CONSTANTS.ACTIONS.CLOSE_DIALOG,
        dialogType
    });
};

const openHintDialog = () => {
    openDialog(
        CONSTANTS.DIALOG_OPTIONS.HINT.URL,
        CONSTANTS.DIALOG_OPTIONS.HINT.WIDTH,
        CONSTANTS.DIALOG_OPTIONS.HINT.HEIGHT
    );
};

const openLocatorDialog = async () => {
    const data = await chrome.storage.local.get('saveFeatureEnabled');

    let height = data.saveFeatureEnabled ?
        CONSTANTS.DIALOG_OPTIONS.LOCATOR.ADVANCED_HEIGHT :
        CONSTANTS.DIALOG_OPTIONS.LOCATOR.HEIGHT;

    openDialog(
        CONSTANTS.DIALOG_OPTIONS.LOCATOR.URL,
        CONSTANTS.DIALOG_OPTIONS.LOCATOR.WIDTH,
        height
    );
};

const openDialog = (url, width, height, isLogInfo) => {
    if (!isLogInfo) {
        dialogTabId = null;
    } else {
        logInfoDialogTabId = null;
    }

    chrome.windows.getCurrent(window => {
        let screenDimensions =
            {
                width: window.width,
                height: window.height
            };
        chrome.windows.create(
            {
                url: url,
                type: 'popup',
                width: width,
                height: height,
                left: Math.round((screenDimensions.width / 2) - (width / 2)),
                top: Math.round((screenDimensions.height / 2) - (height / 2)),
                focused: false
            },
            window => {
                if (!isLogInfo) {
                    dialogTabId = window.tabs[0].id;
                } else {
                    logInfoDialogTabId = window.tabs[0].id;
                }
                focusTab(window.tabs[0].id);
            }
        );
    });
};

const closeDialog = () => {
    if (dialogTabId !== null) {
        chrome.tabs.get(dialogTabId, tab => {
            dialogTabId = null;
            chrome.windows.remove(tab.windowId);
        });
    }
};

const focusTab = tabId => {
    chrome.tabs.get(tabId, (tab) => {
        chrome.windows.update(tab.windowId, {'focused': true});
        chrome.tabs.update(tab.id, {'active': true});
        if (tabId !== logInfoDialogTabId) {
            changeIcon(true);
        }
    });
};

const sendNotification = (message, type) => {
    chrome.tabs.query({active: true}, tabs => {
        chrome.tabs.sendMessage(
            tabs[0].id,
            {
                action: CONSTANTS.ACTIONS.CHECK_IS_FRAMESET
            },
            response => {
                if (response && response.isFrameset) {
                    showMessage(message);
                } else {
                    chrome.tabs.sendMessage(tabs[0].id, {
                        action: CONSTANTS.ACTIONS.SHOW_NOTIFICATION,
                        message,
                        type
                    });
                }
            }
        );
    });
};

const closeNotification = type => {
    chrome.tabs.query({active: true}, tabs => {
        chrome.tabs.sendMessage(tabs[0].id, {
            action: CONSTANTS.ACTIONS.CLOSE_NOTIFICATION,
            type
        });
    });
};

const showMessage = message => {
    if (isNavigatorWebdriver) {
        showChromeNotification(message);
    } else {
        chrome.tabs.query({active: true}, tabs => {
            chrome.tabs.sendMessage(
                tabs[0].id,
                {
                    action: CONSTANTS.ACTIONS.SHOW_ALERT,
                    message
                },
                response => {
                    if (chrome.runtime.lastError) {
                        showChromeNotification(message);
                    }
                }
            );
        });
    }
};

const showChromeNotification = message => {
    chrome.notifications.clear(CONSTANTS.NOTIFICATION_ID, () => {
        chrome.notifications.create(CONSTANTS.NOTIFICATION_ID,
            {
                type: 'basic',
                iconUrl: CONSTANTS.NOTIFICATION_ICON_URL,
                title: CONSTANTS.EXTENSION_NAME,
                message
            });
    });
};

const setConnectionInfo = async data => {
    let savingSettings = {
        host: ApiService.normalizeURL(data.host),
        isBasicAuth: false,
        username: '',
        apikey: data.apiKey,
        auth: data.apiKey,
        saveCssSelector: false
    };
    let templates = formatTemplates(data.templates);

    await chrome.storage.local.set({settings: savingSettings, saveFeatureEnabled: true, templates: templates});
};

const formatTemplates = sourceTemplates => {
    let result = [];
    let templates = sourceTemplates[1];
    templates.forEach(template => {
        let sourceTemplate = template[1];
        result.push({
            name: sourceTemplate.name,
            id: sourceTemplate.id,
            patterns: formatPatterns(sourceTemplate.patterns)
        });
    });
    return result;
};

const formatPatterns = sourcePatterns => {
    let result = [];
    let patterns = sourcePatterns[1];
    patterns.forEach(pattern => {
        let sourcePattern = pattern[1];
        result.push({
            description: sourcePattern.description,
            xpath: sourcePattern.xpath,
            id: sourcePattern.id,
            orderNumber: sourcePattern.orderNumber,
            weight: sourcePattern.weight
        });
    });
    return result;
};

const createLogInfoPopup = () => {
    openDialog(
        CONSTANTS.LOG_INFO.URL,
        CONSTANTS.LOG_INFO.WIDTH,
        CONSTANTS.LOG_INFO.HEIGHT,
        true
    );
};

const sendMessageToLogInfo = message => {
    if (logInfoPort) {
        logInfoPort.postMessage(message);
    }
};

const addLogMenuItem = () => {
    chrome.contextMenus.create({
            id: CONSTANTS.LOG_INFO.MENU_ID,
            title: CONSTANTS.LOG_INFO.MENU_TITLE,
            contexts: ['action']
        },
        () => {
            if (chrome.runtime.lastError) {
                console.debug(`Error creating context menu: ${chrome.runtime.lastError.message}`);
            }
        });
};

const removeLogMenuItem = skipError => {
    chrome.contextMenus.remove(CONSTANTS.LOG_INFO.MENU_ID, () => {
        if (chrome.runtime.lastError) {
            if (!skipError) {
                console.debug(`No existing menu item with ID "${CONSTANTS.LOG_INFO.MENU_ID}" to remove or other error: ${chrome.runtime.lastError.message}`);
            }
        }
    });
};

const createOrUpdateLogMenuItem = () => {
    removeLogMenuItem(true);
    addLogMenuItem();
};
